# Simulate coalescent process with heterochronous sampling times

# Assumptions and modifications
# - simulate a single rate-shift
# - uses phylodyn package of Karcher 2016 et al
# - samples placed isochronously at 0 and tshift
# - altChange gives fraction of fall after shift
# - removed several outputs e.g. tree

# Clean the workspace and console
closeAllConnections()
rm(list=ls())
cat("\014")  
graphics.off()

# Packages for phylodyn
library("sp")
library("devtools")
library("INLA")
library("spam")
library("ape")
library("phylodyn")

# Set working directory to source
this.dir <- dirname(parent.frame(2)$ofile)
setwd(this.dir)

# Function to write simple csv files to correct path
tableWrite <- function(val, name, pathname) {
  # Add path to name
  str0 <- paste(c(pathname, name), collapse = "")
  # Write table
  write.table(val, str0, row.names=FALSE, col.names=FALSE, sep=",")
}


# Main code for heterochronous simulations ----------------------------------------------------------

# Shift time and fraction of change
tshift = 200; fr = 0.5
# Null model probability and weights
P0 = 0.5; wt = c(P0, 1-P0)
# No. trajectories with or without shift
numTraj = 100
# Randomly generate these trajectories
fracs = sample(c(1, fr), numTraj, replace=TRUE, prob = wt)

# Times sequence samples introduced
samp_times = seq(0, tshift, length.out = 2)
# Number of samples introduced
nsamps = 101; samps = rep(nsamps/2, 2)

for (i in 1:numTraj) {
  # Create folder for traj specific results
  trajName = paste(c('shift/', 'shift', i), collapse = '')
  dir.create(file.path(this.dir, trajName))

  # Define rate-shift with population fall
  shift_traj <- function (t, frac = fracs[i], ts = tshift) 
  {
    result = rep(0, length(t))
    result[t <= tshift] <- 1000
    result[t >= tshift] <- frac*1000
    return(result)
  }
  traj = shift_traj

  # Simulate genealogy and get all times
  gene = coalsim(samp_times = samp_times, n_sampled = samps, traj = traj, lower_bound = 10, method = "thin")
  coal_times = gene$coal_times
  coalLin = gene$lineages
  
  # Obtain true trajectory across time
  tmax = max(coal_times)
  t = seq(0, tmax, length=10000)
  y = traj(t)
  
  # Export data for Matlab
  pathf = paste(c(this.dir, '/', trajName, '/'), collapse = "")
  tableWrite(coal_times, 'coaltimes.csv', pathf)
  tableWrite(samp_times, 'samptimes.csv', pathf)
  tableWrite(coalLin, 'coalLin.csv', pathf)
  tableWrite(y, 'trajy.csv', pathf)
  tableWrite(t, 'trajt.csv', pathf)
  tableWrite(samps, 'sampIntro.csv', pathf)
  tableWrite(fracs[i], 'frac.csv', pathf)
}
